// $Id: CRangeSlider.cpp,v 1.6 2007/02/08 21:07:54 paul Exp $

/*
 * All contents of this source code are copyright 2005 Exp Digital Uk.
 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy
 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
 * All content is the Intellectual property of Exp Digital Uk.
 * Certain sections of this code may come from other sources. They are credited where applicable.
 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
 */

#include "CRangeSlider.hpp"
using Exponent::GUI::Controls::CRangeSlider;

//	===========================================================================
EXPONENT_CLASS_IMPLEMENTATION(CRangeSlider, CControl);

//	===========================================================================
CRangeSlider::CRangeSlider(IControlRoot *root, const long uniqueId, const CRect &area, IActionListener *listener) 
			: CControl(root, uniqueId, area, listener)
			, m_dragId(e_notDragging)
			, m_isVertical(true)
			, m_state(CRolloverButton::e_mouseOff)
			, m_minValue(0.0)
			, m_maxValue(1.0)
			, m_lockMax(false)
			, m_lockMin(false)
			, m_jumpToMax(false)
			, m_handleColour(CAlphaColour::CALPHACOLOUR_PURPLE)
			, m_handleOverColour(CAlphaColour::CALPHACOLOUR_ORANGE)
			, m_handleDownColour(CAlphaColour::CALPHACOLOUR_RED)
{
	EXPONENT_CLASS_CONSTRUCTION(CRangeSlider);

	// Are we vertical or horizontal?
	(m_area.getWidth() > m_area.getHeight()) ? m_isVertical = false : m_isVertical = true;

	// Reset the slider area
	m_sliderArea.setOrigin(CPoint(0, 0));

	// The width of the slider relative to its area is always equal, no matter wether horzontal or vertical
	if (m_isVertical)
	{
		m_sliderArea.setWidth(area.getWidth());
		m_sliderArea.setHeight(area.getHeight() / 3);
	}
	else
	{
		m_sliderArea.setWidth(area.getWidth() / 3);
		m_sliderArea.setHeight(area.getHeight());
	}
}

//	===========================================================================
CRangeSlider::~CRangeSlider()
{
	EXPONENT_CLASS_DESTRUCTION(CRangeSlider);
}

//	===========================================================================
void CRangeSlider::setMinimumValue(const double minValue)
{
	if (minValue >= 0.0 && minValue <= 1.0)
	{
		m_minValue = minValue;
		if (m_isVertical)
		{
			m_sliderArea.setHeight(((long)((1.0 - m_minValue) * m_area.getHeight())) - m_sliderArea.getTop());
		}
		else
		{
			const long left = ((long)(m_minValue * m_area.getWidth()));
			m_sliderArea.setLeft(left);
			m_sliderArea.setWidth(m_sliderArea.getRight() - left);
		}
		this->update();
	}
}

//	===========================================================================
void CRangeSlider::setMaximumValue(const double maxValue)
{
	if (maxValue >= 0.0 && maxValue <= 1.0)
	{
		m_maxValue = maxValue;
		if (m_isVertical)
		{
			m_sliderArea.setTop((long)((1.0 - m_maxValue) * m_area.getHeight()));
			m_sliderArea.setHeight(m_area.getHeight() - m_sliderArea.getTop());
		}
		else
		{
			m_sliderArea.setWidth((long)(m_maxValue * m_area.getWidth()));
		}
		this->update();
	}
}

//	===========================================================================
void CRangeSlider::setColours(CAlphaColour frameColour, CAlphaColour backgroundColour, CAlphaColour handleColour, CAlphaColour handleOverColour, CAlphaColour handleDownColour)
{
	this->setDefaultControlColours(backgroundColour, frameColour);
	m_handleColour	   = handleColour;
	m_handleOverColour = handleOverColour;
	m_handleDownColour = handleDownColour;
}

//	===========================================================================
void CRangeSlider::lockPositions(const bool lockMin, const bool lockMax, const double minValue, const double maxValue, const bool jumpToMax)
{
	if (minValue < maxValue)
	{
		this->setMaximumValue(maxValue);
		this->setMinimumValue(minValue);
	}
	m_lockMax = lockMax;
	m_lockMin = lockMin;
	m_jumpToMax = jumpToMax;
	if (m_jumpToMax)
	{
		m_lockMin = true;
		m_lockMax = false;
	}
}

//	===========================================================================
void CRangeSlider::handleLeftButtonDown(CMouseEvent &event)
{
	if (m_jumpToMax)
	{
		m_dragId = e_draggingTop;
		if (m_isVertical)
		{
			// Set the top of the slider
			m_sliderArea.setTop(event.getMousePosition().getYPosition());
			m_sliderArea.setHeight(m_area.getHeight() - event.getMousePosition().getYPosition());

			// Set the value of the slider
			m_maxValue = 1.0 - ((double)(m_sliderArea.getTop()) / (double)(m_area.getHeight()));
			m_minValue = 1.0 - ((double)(m_sliderArea.getBottom()) / (double)(m_area.getHeight()));

			// Notify the listener
			if (m_actionListener)
			{
				m_actionListener->handleActionEvent(CActionEvent(this, event));
			}
		}
		else
		{
			// Set the top of the slider
			m_sliderArea.setLeft(0);
			m_sliderArea.setWidth(event.getMousePosition().getXPosition());

			// Set the value of the slider
			m_maxValue = ((double)(m_sliderArea.getRight()) / (double)(m_area.getWidth()));
			m_minValue = ((double)(m_sliderArea.getLeft())  / (double)(m_area.getWidth()));

			// Notify the listener
			if (m_actionListener)
			{
				m_actionListener->handleActionEvent(CActionEvent(this, event));
			}
		}
		m_rootControl->lockControl(this);
		m_grabOffset = event.getMousePosition();
		m_grabOffset.offset(CPoint(-m_sliderArea.getLeft(), -m_sliderArea.getTop()));
		m_state		 = CRolloverButton::e_mouseDown;
	}
	else
	{
		// Top and bottom drag areas
		CRect topHandle;
		CRect bottomHandle;

		if (m_isVertical)
		{
			topHandle.setRect(0, m_sliderArea.getTop() - 2, m_area.getWidth(), 4);
			bottomHandle.setRect(0, m_sliderArea.getBottom() - 2, m_area.getWidth(), 4);
		}
		else
		{
			topHandle.setRect(m_sliderArea.getRight() - 2, 0, 4, m_area.getHeight());
			bottomHandle.setRect(m_sliderArea.getLeft() - 2, 0, 4, m_area.getHeight());
		}

		// Now we need to find where they clicked
		if (topHandle.pointIsInside(event.getMousePosition()))
		{
			if (!m_lockMax)
			{
				m_dragId = e_draggingTop;
				if (m_isVertical)
				{
					event.getMutableMouse()->setCursor(&CCursor::CCURSOR_UP_DOWN);
				}
				else
				{
					event.getMutableMouse()->setCursor(&CCursor::CCURSOR_LEFT_RIGHT);
				}
			}
			else
			{
				m_dragId = e_notDragging;
			}

			m_rootControl->lockControl(this);
			m_grabOffset = event.getMousePosition();
			m_grabOffset.offset(CPoint(-m_sliderArea.getLeft(), -m_sliderArea.getTop()));
			m_state		 = CRolloverButton::e_mouseDown;
		}
		else if (bottomHandle.pointIsInside(event.getMousePosition()))
		{
			if (!m_lockMin)
			{
				m_dragId = e_draggingBottom;
				if (m_isVertical)
				{
					event.getMutableMouse()->setCursor(&CCursor::CCURSOR_UP_DOWN);
				}
				else
				{
					event.getMutableMouse()->setCursor(&CCursor::CCURSOR_LEFT_RIGHT);
				}
			}
			else
			{
				m_dragId = e_notDragging;
			}

			m_rootControl->lockControl(this);
			m_grabOffset = event.getMousePosition();
			m_grabOffset.offset(CPoint(-m_sliderArea.getLeft(), -m_sliderArea.getTop()));
			m_state		 = CRolloverButton::e_mouseDown;
		}
		else if (m_sliderArea.pointIsInside(event.getMousePosition()) && !m_lockMax && !m_lockMin)
		{
			event.getMutableMouse()->setCursor(&CCursor::CCURSOR_ARROW);
			m_dragId = e_draggingHandle;
			
			m_rootControl->lockControl(this);
			m_grabOffset = event.getMousePosition();
			m_grabOffset.offset(CPoint(-m_sliderArea.getLeft(), -m_sliderArea.getTop()));
			m_state		 = CRolloverButton::e_mouseDown;
		}
		else if (!m_lockMax && !m_lockMin)
		{
			event.getMutableMouse()->setCursor(&CCursor::CCURSOR_ARROW);
			// Jump the slider to the new position.
			// When we jump the aim is to make the mid point of the slider
			// the point where they just clicked
			// This point may be offset if for example, they click right at the edge of the 
			// control
			const CPoint clickPoint = event.getMousePosition();

			// Adjust based on direction
			if (m_isVertical)
			{
				// Get the offset position
				long halfHeight = m_sliderArea.getHeight() / 2;

				// Store the bottom and check its range
				long bottom = clickPoint.getYPosition() + halfHeight;
				if (bottom > m_area.getHeight())
				{
					bottom -= (bottom - m_area.getHeight());
				}

				// Store the top and check its range
				long top = bottom - m_sliderArea.getHeight();
				if (top < 0)
				{
					top = 0;
				}

				// Finally set the position
				m_sliderArea.setTop(top);
			}
			else
			{
				// Get the offset position
				long halfWidth = m_sliderArea.getWidth() / 2;

				// Store the bottom and check its range
				long right = clickPoint.getXPosition() + halfWidth;
				if (right > m_area.getWidth())
				{
					right -= (right - m_area.getWidth());
				}

				// Store the top and check its range
				long left = right - m_sliderArea.getWidth();
				if (left < 0)
				{
					left = 0;
				}

				// Finally set the position
				m_sliderArea.setLeft(left);
			}
			m_dragId = e_draggingHandle;
		}
	}

	// Redraw
	this->update();
}

//	===========================================================================
void CRangeSlider::handleLeftButtonUp(CMouseEvent &event)
{
	if (!m_sliderArea.pointIsInside(event.getMousePosition()))
	{
		m_rootControl->unlockControl();
		m_state = CRolloverButton::e_mouseOff;
	}
	else
	{
		m_state = CRolloverButton::e_mouseOver;
	}
	//event.getMutableMouse()->setCursor(&CCursor::CCURSOR_ARROW);
	m_dragId = e_notDragging;
	m_grabOffset.setPoint(0, 0);

	// Notify the listener
	if (m_actionListener)
	{
		m_actionListener->handleActionEvent(CActionEvent(this, event));
	}

	this->update();
}

//	===========================================================================
void CRangeSlider::handleMouseScroll(CMouseEvent &event)
{
	if (m_isVertical)
	{
		// Get the amount to shift by
		long offset = 0;
		if (event.getWheelMovementAmount() < 0) 
		{
			(event.isShiftDown()) ? offset = 1 : offset = 5;
		}
		else
		{
			(event.isShiftDown()) ? offset = -1 : offset = -5;
		}

		// Set the position
		m_sliderArea.setTop(CBounds::ensureRange(m_sliderArea.getTop() + offset, 0, m_area.getHeight() - m_sliderArea.getHeight()));

		// Set the value of the slider
		m_maxValue = 1.0 - ((double)(m_sliderArea.getTop()) / (double)(m_area.getHeight()));
		m_minValue = 1.0 - ((double)(m_sliderArea.getBottom()) / (double)(m_area.getHeight()));

		// Notify the listener
		if (m_actionListener)
		{
			m_actionListener->handleActionEvent(CActionEvent(this, event));
		}
	}
	else
	{
		// Get the amount to shift by
		long offset = 0;
		if (event.getWheelMovementAmount() < 0) 
		{
			(event.isShiftDown()) ? offset = -1 : offset = -5;
		}
		else
		{
			(event.isShiftDown()) ? offset = 1 : offset = 5;
		}

		// Set the position
		m_sliderArea.setLeft(CBounds::ensureRange(m_sliderArea.getLeft() + offset, 0, m_area.getWidth() - m_sliderArea.getWidth()));

		// Set the value of the slider
		m_maxValue = ((double)(m_sliderArea.getRight()) / (double)(m_area.getWidth()));
		m_minValue = ((double)(m_sliderArea.getLeft())  / (double)(m_area.getWidth()));

		// Notify the listener
		if (m_actionListener)
		{
			m_actionListener->handleActionEvent(CActionEvent(this, event));
		}
	}

	this->checkAndLock(event.getMousePosition());
	this->update();
}

//	===========================================================================
void CRangeSlider::handleMouseMovement(CMouseEvent &event)
{
	switch(m_dragId)
	{
		case e_notDragging:	
			this->handleMovement(event);
		break;
		case e_draggingTop:
			if (!this->handleTopMovement(event))
			{
				return;
			}
		break;
		case e_draggingBottom:
			if (!this->handleBottomMovement(event))
			{
				return;
			}
		break;
		case e_draggingHandle:
			if (!this->handleBarMovement(event))
			{
				return;
			}
		break;
	}
	this->update();
}

//	===========================================================================
void CRangeSlider::handleMouseLeavingArea(CMouseEvent &event)
{
	if (m_dragId == e_notDragging)
	{
		m_state = CRolloverButton::e_mouseOff;
		this->update();
	}
}

//	===========================================================================
void CRangeSlider::drawControl(CGraphics &graphics)
{
	// First check if we can allow the standard handler to draw the disabled control
	if (!this->drawEnabledControl(graphics))
	{
		return;
	}

	// Draw the background
	this->drawPrimaryImage(graphics, m_doDefaultDrawing);

	// Now draw the slider
	CAlphaColour theColour;
	switch(m_state)
	{
		case CRolloverButton::e_mouseOff:	theColour = m_handleColour;			break;
		case CRolloverButton::e_mouseOver:	theColour = m_handleOverColour;		break;
		case CRolloverButton::e_mouseDown:	theColour = m_handleDownColour;		break;
	}

	// Draw the box in
	graphics.getMutableBrush()->setColour(theColour);
	graphics.fillRectangle(m_sliderArea);
	
	// Draw the surround in
	graphics.getMutablePen()->setColour(CAlphaColour::CALPHACOLOUR_BLACK);
	graphics.drawRectangle(m_sliderArea);
}

//	===========================================================================
void CRangeSlider::checkAndLock(const CPoint &point)
{
	// Top and bottom drag areas
	CRect topHandle;
	CRect bottomHandle;

	if (m_isVertical)
	{
		topHandle.setRect(0, m_sliderArea.getTop() - 2, m_area.getWidth(), 4);
		bottomHandle.setRect(0, m_sliderArea.getBottom() - 2, m_area.getWidth(), 4);
	}
	else
	{
		topHandle.setRect(m_sliderArea.getRight() - 2, 0, 4, m_area.getHeight());
		bottomHandle.setRect(m_sliderArea.getLeft() - 2, 0, 4, m_area.getHeight());
	}

	// Now we need to find where they clicked
	if (topHandle.pointIsInside(point) && !m_lockMax)
	{
		m_state = CRolloverButton::e_mouseOver;
		m_rootControl->lockControl(this);
	}
	else if (bottomHandle.pointIsInside(point) && !m_lockMin)
	{
		m_state = CRolloverButton::e_mouseOver;
		m_rootControl->lockControl(this);
	}
	else
	{
		m_state = CRolloverButton::e_mouseOff;
		m_rootControl->unlockControl();
	}
}

//	===========================================================================
bool CRangeSlider::handleTopMovement(CMouseEvent &event)
{
	if (m_isVertical)
	{
		event.getMutableMouse()->setCursor(&CCursor::CCURSOR_UP_DOWN);
		const long bottom = m_sliderArea.getBottom();
		long yPos         = event.getMousePosition().getYPosition();

		// Check that its in range
		if (yPos < 0)
		{
			yPos = 0;
		}

		if (yPos > bottom)
		{
			yPos = bottom;
		}

		if (yPos > m_area.getHeight())
		{
			yPos = m_area.getHeight();
		}
		
		// Set the slider area
		m_sliderArea.setTop(yPos);
		m_sliderArea.setHeight(bottom - yPos);

		// Store the values
		m_maxValue = 1.0 - ((double)(m_sliderArea.getTop())  / (double)(m_area.getHeight()));

		// Notify the listener
		if (m_actionListener)
		{
			m_actionListener->handleActionEvent(CActionEvent(this, event));
		}
	}
	else
	{
		event.getMutableMouse()->setCursor(&CCursor::CCURSOR_LEFT_RIGHT);
		const long left = m_sliderArea.getLeft();
		long xPos       = event.getMousePosition().getXPosition();

		// Check that its in range
		if (xPos < 0)
		{
			xPos = 0;
		}

		if (xPos < left + 1)
		{
			xPos = left + 1;
		}

		if (xPos > m_area.getWidth())
		{
			xPos = m_area.getWidth();
		}
		
		// Set the slider area
		m_sliderArea.setWidth(xPos - left);

		// Store the values
		m_maxValue = ((double)(m_sliderArea.getRight())  / (double)(m_area.getWidth()));

		// Notify the listener
		if (m_actionListener)
		{
			m_actionListener->handleActionEvent(CActionEvent(this, event));
		}
	}
	return true;
}

//	===========================================================================
bool CRangeSlider::handleBottomMovement(CMouseEvent &event)
{
	if (m_isVertical)
	{
		event.getMutableMouse()->setCursor(&CCursor::CCURSOR_UP_DOWN);
		const long top  = m_sliderArea.getTop();
		long yPos   = event.getMousePosition().getYPosition();

		// Check that its in range
		if (yPos < 0)
		{
			yPos = 0;
		}

		if (yPos < top + 1)
		{
			yPos = top + 1;
		}

		if (yPos > m_area.getHeight())
		{
			yPos = m_area.getHeight();
		}

		// Set the height of the slider
		m_sliderArea.setHeight(yPos - top);

		// Set the values
		m_minValue = 1.0 - ((double)(m_sliderArea.getBottom())  / (double)(m_area.getHeight()));

		// Notify the listener
		if (m_actionListener)
		{
			m_actionListener->handleActionEvent(CActionEvent(this, event));
		}
	}
	else
	{
		event.getMutableMouse()->setCursor(&CCursor::CCURSOR_LEFT_RIGHT);
		const long right  = m_sliderArea.getRight();
		long xPos         = event.getMousePosition().getXPosition();

		// Check that its in range
		if (xPos < 0)
		{
			xPos = 0;
		}

		if (xPos > right - 1)
		{
			xPos = right - 1;
		}

		if (xPos > m_area.getWidth())
		{
			xPos = m_area.getWidth();
		}

		// Set the height of the slider
		m_sliderArea.setLeft(xPos);
		m_sliderArea.setWidth(right - xPos);

		// Set the values
		m_minValue = ((double)(m_sliderArea.getLeft())  / (double)(m_area.getWidth()));

		// Notify the listener
		if (m_actionListener)
		{
			m_actionListener->handleActionEvent(CActionEvent(this, event));
		}
	}
	return true;
}

//	===========================================================================
void CRangeSlider::handleMovement(CMouseEvent &event)
{
	// Capture control as needed
	this->checkAndLock(event.getMousePosition());

	// Top and bottom drag areas
	CRect topHandle;
	CRect bottomHandle;

	if (m_isVertical)
	{
		topHandle.setRect(0, m_sliderArea.getTop() - 2, m_area.getWidth(), 4);
		bottomHandle.setRect(0, m_sliderArea.getBottom() - 2, m_area.getWidth(), 4);
	}
	else
	{
		topHandle.setRect(m_sliderArea.getRight() - 2, 0, 4, m_area.getHeight());
		bottomHandle.setRect(m_sliderArea.getLeft() - 2, 0, 4, m_area.getHeight());
	}

	// Now we need to find where they clicked
	if (topHandle.pointIsInside(event.getMousePosition()) && !m_lockMax)
	{
		if (m_isVertical)
		{
			event.getMutableMouse()->setCursor(&CCursor::CCURSOR_UP_DOWN);
		}
		else
		{
			event.getMutableMouse()->setCursor(&CCursor::CCURSOR_LEFT_RIGHT);
		}
	}
	else if (bottomHandle.pointIsInside(event.getMousePosition()) && !m_lockMin)
	{
		if (m_isVertical)
		{
			event.getMutableMouse()->setCursor(&CCursor::CCURSOR_UP_DOWN);
		}
		else
		{
			event.getMutableMouse()->setCursor(&CCursor::CCURSOR_LEFT_RIGHT);
		}
	}
	else
	{
		event.getMutableMouse()->setCursor(&CCursor::CCURSOR_ARROW);
	}
}

//	===========================================================================
bool CRangeSlider::handleBarMovement(CMouseEvent &event)
{
	event.getMutableMouse()->setCursor(&CCursor::CCURSOR_ARROW);
	if (m_isVertical)
	{
		// Get the position for future use
		long yPos = event.getMousePosition().getYPosition();

		// Check that its in range
		if (yPos < 0)
		{
			yPos = 0;
		}

		// Set the top
		m_sliderArea.setTop(CBounds::ensureRange(yPos - m_grabOffset.getYPosition(), 0, m_area.getHeight() - m_sliderArea.getHeight()));

		// Store the value
		m_maxValue = 1.0 - ((double)(m_sliderArea.getTop())  / (double)(m_area.getHeight()));
		m_minValue = 1.0 - ((double)(m_sliderArea.getBottom())  / (double)(m_area.getHeight()));

		// Notify the listener
		if (m_actionListener)
		{
			m_actionListener->handleActionEvent(CActionEvent(this, event));
		}
	}
	else
	{
		// Get the position for future use
		long xPos = event.getMousePosition().getXPosition();

		// Check that its in range
		if (xPos < 0)
		{
			xPos = 0;
		}

		// Set the top
		m_sliderArea.setLeft(CBounds::ensureRange(xPos - m_grabOffset.getXPosition(), 0, m_area.getWidth() - m_sliderArea.getWidth()));

		// Store the value
		m_maxValue = ((double)(m_sliderArea.getRight())  / (double)(m_area.getWidth()));
		m_minValue = ((double)(m_sliderArea.getLeft())  / (double)(m_area.getWidth()));

		// Notify the listener
		if (m_actionListener)
		{
			m_actionListener->handleActionEvent(CActionEvent(this, event));
		}
	}
	return true;
}